/*-
 * Copyright (c) 2004-2009, Jilles Tjoelker
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with
 * or without modification, are permitted provided that the
 * following conditions are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 * 2. Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__FBSDID("$FreeBSD$");

#include <sys/types.h>
#include <sys/event.h>
#include <sys/time.h>
#include <sys/wait.h>

#include <err.h>
#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sysexits.h>
#include <unistd.h>

#ifdef SHELL
#define main pwaitcmd
#include "bltin/bltin.h"
#include "helpers.h"
#endif

static void
usage(void)
{

	fprintf(stderr, "usage: pwait [-t timeout] [-ov] pid ...\n");
	exit(EX_USAGE);
}

/*
 * pwait - wait for processes to terminate
 */
int
main(int argc, char *argv[])
{
#ifdef SHELL
	struct sigdata info_oact, alrm_oact;
	int pushed_alarm;
#endif
	struct itimerval itv;
	struct kevent *e;
	int oflag, tflag, verbose;
	int i, kq, n, nleft, opt, status;
	long pid;
	char *end, *s;
	double timeout;
	pid_t me;

	oflag = 0;
	tflag = 0;
	verbose = 0;
	memset(&itv, 0, sizeof(itv));

#ifdef SHELL
	pushed_alarm = 0;
#endif
	while ((opt = getopt(argc, argv, "ot:v")) != -1) {
		switch (opt) {
		case 'o':
			oflag = 1;
			break;
		case 't':
			tflag = 1;
			errno = 0;
			timeout = strtod(optarg, &end);
			if (end == optarg || errno == ERANGE || timeout < 0) {
				errx(EX_DATAERR, "timeout value");
			}
			switch(*end) {
			case 0:
			case 's':
				break;
			case 'h':
				timeout *= 60;
				/* FALLTHROUGH */
			case 'm':
				timeout *= 60;
				break;
			default:
				errx(EX_DATAERR, "timeout unit");
			}
			if (timeout > 100000000L) {
				errx(EX_DATAERR, "timeout value");
			}
			itv.it_value.tv_sec = (time_t)timeout;
			timeout -= (time_t)timeout;
			itv.it_value.tv_usec =
			    (suseconds_t)(timeout * 1000000UL);
			break;
		case 'v':
			verbose = 1;
			break;
		default:
			usage();
			/* NOTREACHED */
		}
	}

	argc -= optind;
	argv += optind;

	if (argc == 0) {
		usage();
	}

#ifndef SHELL
	me = getpid();
#else
	me = -1;
#endif

#ifdef SHELL
	INTOFF;
	trap_push(SIGINFO, &info_oact);
#endif
	kq = kqueue();
	if (kq == -1) {
#ifdef SHELL
		trap_pop(SIGINFO, &info_oact);
		INTON;
#endif
		err(EX_OSERR, "kqueue");
	}

	e = malloc((argc + tflag) * sizeof(struct kevent));
	if (e == NULL) {
#ifdef SHELL
		close(kq);
		trap_pop(SIGINFO, &info_oact);
		INTON;
#endif
		err(EX_OSERR, "malloc");
	}
	nleft = 0;
	for (n = 0; n < argc; n++) {
		s = argv[n];
		/* Undocumented Solaris compat */
		if (!strncmp(s, "/proc/", 6)) {
			s += 6;
		}
		errno = 0;
		pid = strtol(s, &end, 10);
		if (pid < 0 || *end != '\0' || errno != 0) {
			warnx("%s: bad process id", s);
			continue;
		}
		if (pid == me) {
			warnx("%s: ignoring own process id", s);
			continue;
		}
		for (i = 0; i < nleft; i++) {
			if (e[i].ident == (uintptr_t)pid) {
				break;
			}
		}
		if (i < nleft) {
			/* Duplicate. */
			continue;
		}
		EV_SET(e + nleft, pid, EVFILT_PROC, EV_ADD, NOTE_EXIT, 0, NULL);
		if (kevent(kq, e + nleft, 1, NULL, 0, NULL) == -1) {
			warn("%ld", pid);
			if (oflag) {
#ifdef SHELL
				close(kq);
				free(e);
				trap_pop(SIGINFO, &info_oact);
				INTON;
#endif
				exit(EX_OK);
			}
		} else {
			nleft++;
		}
	}

	if (nleft > 0 && tflag) {
		/*
		 * Explicitly detect SIGALRM so that an exit status of 124
		 * can be returned rather than 142.
		 */
		EV_SET(e + nleft, SIGALRM, EVFILT_SIGNAL, EV_ADD, 0, 0, NULL);
		if (kevent(kq, e + nleft, 1, NULL, 0, NULL) == -1) {
#ifdef SHELL
			close(kq);
			free(e);
			trap_pop(SIGINFO, &info_oact);
			INTON;
#endif
			err(EX_OSERR, "kevent");
		}
		/* Ignore SIGALRM to not interrupt kevent(2). */
#ifdef SHELL
		trap_push(SIGALRM, &alrm_oact);
		pushed_alarm = 1;
#else
		signal(SIGALRM, SIG_IGN);
#endif
		if (setitimer(ITIMER_REAL, &itv, NULL) == -1) {
#ifdef SHELL
			close(kq);
			free(e);
			trap_pop(SIGINFO, &info_oact);
			trap_pop(SIGALRM, &alrm_oact);
			INTON;
#endif
			err(EX_OSERR, "setitimer");
		}
	}
	while (nleft > 0) {
		n = kevent(kq, NULL, 0, e, nleft + tflag, NULL);
		if (n == -1) {
#ifdef SHELL
			close(kq);
			free(e);
			trap_pop(SIGINFO, &info_oact);
			if (tflag && pushed_alarm) {
				alarm(0);
				trap_pop(SIGALRM, &alrm_oact);
			}
			INTON;
#endif
			err(EX_OSERR, "kevent");
		}
		for (i = 0; i < n; i++) {
			if (e[i].filter == EVFILT_SIGNAL) {
				if (verbose) {
					printf("timeout\n");
				}
#ifdef SHELL
				close(kq);
				free(e);
				trap_pop(SIGINFO, &info_oact);
				if (tflag && pushed_alarm) {
					alarm(0);
					trap_pop(SIGALRM, &alrm_oact);
				}
				INTON;
#endif
				exit(124);
			}
			if (verbose) {
				status = e[i].data;
				if (WIFEXITED(status)) {
					printf("%ld: exited with status %d.\n",
					    (long)e[i].ident,
					    WEXITSTATUS(status));
				} else if (WIFSIGNALED(status)) {
					printf("%ld: killed by signal %d.\n",
					    (long)e[i].ident,
					    WTERMSIG(status));
				} else {
					printf("%ld: terminated.\n",
					    (long)e[i].ident);
				}
			}
			if (oflag) {
#ifdef SHELL
				close(kq);
				free(e);
				trap_pop(SIGINFO, &info_oact);
				if (tflag && pushed_alarm) {
					alarm(0);
					trap_pop(SIGALRM, &alrm_oact);
				}
				INTON;
#endif
				exit(EX_OK);
			}
			--nleft;
		}
	}

#ifdef SHELL
	close(kq);
	free(e);
	trap_pop(SIGINFO, &info_oact);
	if (tflag && pushed_alarm) {
		alarm(0);
		trap_pop(SIGALRM, &alrm_oact);
	}
	INTON;
#endif
	exit(EX_OK);
}
